<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Models\Setting;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Livewire\WithFileUploads;

class Settings extends Component
{
    use WithFileUploads;

    public $settings = [];
    public $brand_logo; // For upload
    public $hero_slides = []; // For upload

    public function mount()
    {
        $this->settings = Setting::pluck('value', 'key')->toArray();
    }

    public function updateSetting($key, $value)
    {
        Setting::updateOrCreate(['key' => $key], ['value' => $value]);
        $this->settings[$key] = $value;
        Cache::forget('settings'); // Clear cache if you use it (not used yet but good practice)
        
        // Re-share to view? in Livewire we might need to dispatch an event or just refresh page. 
        // For now simple update is enough.
        $this->dispatch('settings-updated');
    }

    public function updatedBrandLogo()
    {
        $this->validate([
            'brand_logo' => 'image|max:1024', // 1MB Max
        ]);

        $path = $this->brand_logo->store('settings', 'public');
        $url = Storage::url($path);
        
        $this->updateSetting('brand_logo', $url);
        $this->brand_logo = null; // Reset
    }

    public function updatedHeroSlides()
    {
        $this->validate([
            'hero_slides.*' => 'image|max:15360', // 15MB Max per file
        ]);

        // Decode existing slides
        $currentSlides = json_decode($this->settings['hero_slides'] ?? '[]', true);
        $currentCount = count($currentSlides);
        $availableSlots = 5 - $currentCount;

        // Check if already at max
        if ($availableSlots <= 0) {
            session()->flash('hero_error', 'Maximum 5 slides allowed. Please remove some slides first.');
            $this->hero_slides = [];
            return;
        }

        // Only process up to available slots
        $filesToProcess = array_slice($this->hero_slides, 0, $availableSlots);
        $skippedCount = count($this->hero_slides) - count($filesToProcess);
        
        $newSlides = [];
        foreach ($filesToProcess as $slide) {
            $path = $slide->store('hero', 'public');
            $url = Storage::url($path);
            $newSlides[] = [
                'image' => $url,
                'title' => 'Explore Bali',
                'subtitle' => 'Premium Vehicles for Rent'
            ];
        }

        // Merge with existing
        $merged = array_merge($currentSlides, $newSlides);
        
        // Save as JSON
        $this->updateSetting('hero_slides', json_encode($merged));
        
        if ($skippedCount > 0) {
            session()->flash('hero_warning', "Only {$availableSlots} slot(s) available. {$skippedCount} image(s) were skipped.");
        } else {
            session()->flash('hero_success', count($newSlides) . ' slide(s) uploaded successfully.');
        }
        
        $this->hero_slides = []; // Reset
    }
    
    public function updateSlideText($index, $field, $value)
    {
        $slides = json_decode($this->settings['hero_slides'] ?? '[]', true);
        if (isset($slides[$index])) {
            $slides[$index][$field] = $value;
            $this->updateSetting('hero_slides', json_encode($slides));
        }
    }
    
    public function moveSlide($index, $direction)
    {
        $slides = json_decode($this->settings['hero_slides'] ?? '[]', true);
        $newIndex = $direction === 'up' ? $index - 1 : $index + 1;
        
        if ($newIndex >= 0 && $newIndex < count($slides)) {
            // Swap positions
            $temp = $slides[$index];
            $slides[$index] = $slides[$newIndex];
            $slides[$newIndex] = $temp;
            
            $this->updateSetting('hero_slides', json_encode($slides));
        }
    }

    public function removeSlide($index)
    {
        $slides = json_decode($this->settings['hero_slides'] ?? '[]', true);
        if (isset($slides[$index])) {
            array_splice($slides, $index, 1);
            $this->updateSetting('hero_slides', json_encode($slides));
        }
    }

    public function toggle($key)
    {
        $currentValue = $this->settings[$key] ?? '0';
        $newValue = $currentValue == '1' ? '0' : '1';
        
        $this->updateSetting($key, $newValue);
    }

    public function render()
    {
        return view('livewire.admin.settings')->layout('layouts.admin');
    }
}
