<?php

namespace App\Livewire\Admin\Cars;

use App\Models\Car;
use App\Models\CarBrand;
use Livewire\Component;
use Livewire\WithFileUploads;

class Edit extends Component
{
    use WithFileUploads;

    public Car $car;

    public $name;
    public $description;
    public $brand_id;
    public $year;
    public $seats;
    public $transmission;
    public $fuel_type;
    public $price_daily;
    public $price_weekly;
    public $price_monthly;
    public $new_main_image;
    public $images = []; // Existing gallery
    public $new_images = []; // New uploads
    public $status;

    public function mount(Car $car)
    {
        $this->car = $car;
        $this->name = $car->name;
        $this->description = $car->description;
        $this->brand_id = $car->brand_id;
        $this->year = $car->year;
        $this->seats = $car->seats;
        $this->transmission = $car->transmission;
        $this->fuel_type = $car->fuel_type;
        $this->price_daily = $car->price_daily;
        $this->price_weekly = $car->price_weekly;
        $this->price_monthly = $car->price_monthly;
        $this->images = $car->images ?? [];
        $this->status = $car->status;
    }

    public function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'brand_id' => 'required|exists:car_brands,id',
            'year' => 'required|integer|min:2000|max:' . (date('Y') + 1),
            'seats' => 'required|integer|min:1',
            'transmission' => 'required|in:Automatic,Manual',
            'fuel_type' => 'required|string',
            'price_daily' => 'required|numeric|min:0',
            'price_weekly' => 'nullable|numeric|min:0',
            'price_monthly' => 'nullable|numeric|min:0',
            'new_main_image' => 'nullable|image|max:10240',
            'new_images.*' => 'image|max:10240',
            'status' => 'required|in:available,booked,maintenance',
        ];
    }

    public function removeImage($index)
    {
        if (isset($this->images[$index])) {
            unset($this->images[$index]);
            $this->images = array_values($this->images); // Reindex
        }
    }

    public function save()
    {
        $validatedData = $this->validate();

        if ($this->new_main_image) {
            $validatedData['main_image'] = $this->new_main_image->store('cars', 'public');
        } else {
            $validatedData['main_image'] = $this->car->main_image;
        }

        // Handle Gallery
        $currentImages = $this->images; // Start with existing (after deletions)
        
        if ($this->new_images) {
            foreach ($this->new_images as $image) {
                $currentImages[] = $image->store('cars/gallery', 'public');
            }
        }
        
        $validatedData['images'] = $currentImages;

        unset($validatedData['new_main_image']);
        unset($validatedData['new_images']);

        $this->car->update($validatedData);

        session()->flash('success', 'Car updated successfully.');

        return redirect()->route('admin.cars.index');
    }

    public function render()
    {
        return view('livewire.admin.cars.edit', [
            'brands' => CarBrand::all()
        ])->layout('layouts.admin');
    }
}
