<?php

namespace App\Livewire\Admin\Cars;

use App\Models\Car;
use App\Models\CarBrand;
use Livewire\Component;
use Livewire\WithFileUploads;

class Create extends Component
{
    use WithFileUploads;

    public $name;
    public $description;
    public $brand_id;
    public $year;
    public $seats = 4;
    public $transmission = 'Automatic';
    public $fuel_type = 'Petrol';
    public $price_daily;
    public $price_weekly;
    public $price_monthly;
    public $main_image;
    public $images = []; // Multi-image gallery
    public $status = 'available';

    public function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'brand_id' => 'required|exists:car_brands,id',
            'year' => 'required|integer|min:2000|max:' . (date('Y') + 1),
            'seats' => 'required|integer|min:1',
            'transmission' => 'required|in:Automatic,Manual',
            'fuel_type' => 'required|string',
            'price_daily' => 'required|numeric|min:0',
            'price_weekly' => 'nullable|numeric|min:0',
            'price_monthly' => 'nullable|numeric|min:0',
            'main_image' => 'required|image|max:10240', // 10MB Max
            'images.*' => 'image|max:10240', // Gallery validation
            'status' => 'required|in:available,booked,maintenance',
        ];
    }

    public function save()
    {
        $validatedData = $this->validate();

        if ($this->main_image) {
            $validatedData['main_image'] = $this->main_image->store('cars', 'public');
        }

        // Handle Gallery
        if ($this->images) {
            $galleryPaths = [];
            foreach ($this->images as $image) {
                $galleryPaths[] = $image->store('cars/gallery', 'public');
            }
            $validatedData['images'] = $galleryPaths; // Eloquent casts to JSON
        }

        Car::create($validatedData);

        session()->flash('success', 'Car created successfully.');

        return redirect()->route('admin.cars.index');
    }

    public function render()
    {
        return view('livewire.admin.cars.create', [
            'brands' => CarBrand::all()
        ])->layout('layouts.admin');
    }
}
