<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\CarBrand;
use App\Models\MotorBrand;
use Livewire\Attributes\Layout;

#[Layout('layouts.admin')]
class Brands extends Component
{
    use WithFileUploads;

    public $activeTab = 'cars'; // 'cars' or 'motors'
    
    // Form fields
    public $name;
    public $logo;
    public $temp_logo; // For previewing new upload
    
    // Modal state
    public $showModal = false;
    public $isEditing = false;
    public $editId = null;

    public function switchTab($tab)
    {
        $this->activeTab = $tab;
        $this->resetForm();
    }

    public function create()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    public function edit($id)
    {
        $this->resetForm();
        $this->isEditing = true;
        $this->editId = $id;
        
        $model = $this->activeTab === 'cars' ? CarBrand::find($id) : MotorBrand::find($id);
        
        if ($model) {
            $this->name = $model->name;
            $this->logo = $model->logo; // Existing logo path
            $this->showModal = true;
        }
    }

    public function save()
    {
        $rules = [
            'name' => 'required|string|max:255|unique:' . ($this->activeTab === 'cars' ? 'car_brands' : 'motor_brands') . ',name,' . $this->editId,
            'temp_logo' => 'nullable|image|max:1024', // 1MB Max
        ];

        $this->validate($rules);

        $data = [
            'name' => $this->name,
        ];

        // Handle Image Upload
        if ($this->temp_logo) {
            $path = $this->temp_logo->store('brands', 'public');
            $data['logo'] = 'storage/' . $path;
        }

        if ($this->isEditing) {
            $model = $this->activeTab === 'cars' ? CarBrand::find($this->editId) : MotorBrand::find($this->editId);
            $model->update($data);
            session()->flash('message', 'Brand updated successfully.');
        } else {
            if ($this->activeTab === 'cars') {
                CarBrand::create($data);
            } else {
                MotorBrand::create($data);
            }
            session()->flash('message', 'Brand created successfully.');
        }

        $this->showModal = false;
        $this->resetForm();
    }

    public function delete($id)
    {
        $model = $this->activeTab === 'cars' ? CarBrand::find($id) : MotorBrand::find($id);
        
        if ($model) {
            $model->delete();
            session()->flash('message', 'Brand deleted successfully.');
        }
    }

    public function resetForm()
    {
        $this->name = '';
        $this->logo = null;
        $this->temp_logo = null;
        $this->isEditing = false;
        $this->editId = null;
        $this->resetErrorBag();
    }

    public function render()
    {
        $brands = $this->activeTab === 'cars' 
            ? CarBrand::orderBy('name')->get() 
            : MotorBrand::orderBy('name')->get();

        return view('livewire.admin.brands', [
            'brands' => $brands
        ]);
    }
}
