<?php

namespace App\Livewire\Admin\Articles;

use App\Models\Article;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Layout;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

#[Layout('layouts.admin')]
class Edit extends Component
{
    use WithFileUploads;

    public Article $article;
    public $title;
    public $content;
    public $image;
    public $new_image;
    public $status;

    public function mount(Article $article)
    {
        $this->article = $article;
        $this->title = $article->title;
        $this->content = $article->content;
        $this->image = $article->image;
        $this->status = $article->status;
    }

    public function save()
    {
        $this->validate([
            'title' => 'required|min:3',
            'content' => 'required|min:10',
            'new_image' => 'nullable|image|max:2048',
            'status' => 'required|in:draft,published',
        ]);

        $imagePath = $this->image;
        if ($this->new_image) {
            if ($this->image && Storage::disk('public')->exists($this->image)) {
                Storage::disk('public')->delete($this->image);
            }
            $imagePath = $this->new_image->store('articles', 'public');
        }

        $this->article->update([
            'title' => $this->title,
            'slug' => Str::slug($this->title) . '-' . uniqid(), 
            'content' => $this->content,
            'image' => $imagePath,
            'status' => $this->status,
            'published_at' => $this->status === 'published' && !$this->article->published_at ? now() : $this->article->published_at,
        ]);

        return redirect()->route('admin.articles.index')->with('success', 'Article updated successfully.');
    }

    public function render()
    {
        return view('livewire.admin.articles.edit');
    }
}
